<?php
/* --------------------------------------------------------------
   LanguagesLoader.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Layout\Smarty\Loaders;

use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Core\Language\Language;
use Gambio\Core\Language\LanguageService;
use Smarty;

/**
 * Class LanguagesLoader
 *
 * @package Gambio\Admin\Layout\Smarty\Loaders
 */
class LanguagesLoader implements LoaderInterface
{
    /**
     * @var LanguageService
     */
    private $service;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    
    /**
     * LanguagesLoader constructor.
     *
     * @param LanguageService $repository
     * @param UserPreferences $userPreferences
     */
    public function __construct(LanguageService $repository, UserPreferences $userPreferences)
    {
        $this->service         = $repository;
        $this->userPreferences = $userPreferences;
    }
    
    
    /**
     * @inheritDoc
     */
    public function load(Smarty $smarty): void
    {
        $currentLanguageCode = $this->service->getLanguageById($this->userPreferences->languageId())->code();
        
        $smarty->assign('language_code', $currentLanguageCode);
        $smarty->assign('languages', $this->getAvailableLanguagesData());
        $smarty->assign('admin_languages', $this->getAvailableAdminLanguagesData());
    }
    
    
    /**
     * @return array
     */
    private function getAvailableLanguagesData(): array
    {
        $languagesData = [];
        
        /** @var Language $language */
        foreach ($this->service->getAvailableLanguages() as $language) {
            $languagesData[] = [
                'id'   => $language->id(),
                'code' => $language->code(),
                'name' => $language->name(),
            ];
        }
        
        return $languagesData;
    }
    
    
    /**
     * @return array
     */
    private function getAvailableAdminLanguagesData(): array
    {
        $languagesData = [];
        
        /** @var Language $language */
        foreach ($this->service->getAvailableAdminLanguages() as $language) {
            $languagesData[] = [
                'id'   => $language->id(),
                'code' => $language->code(),
                'name' => $language->name(),
            ];
        }
        
        return $languagesData;
    }
}